from PyQt5.QtCore import QRect
import os
import json
import krita

def parse_layer_name(name):
    """Parses a Krita layer name to extract hierarchy, options, and special flags."""
    if name.startswith('--'):
        return None

    info = {
        'path': [],
        'options': {},
        'is_static': False,
        'is_pos': False,
        'is_root_pos': False,
        'original_name': name
    }

    clean_name = name.strip()

    if clean_name == 'root pos':
        info['is_root_pos'] = True
        info['path'] = ['root']
        return info

    parts_str = clean_name
    options_str = ''
    if ' o ' in clean_name:
        parts_str, options_str = clean_name.split(' o ', 1)

    if options_str:
        for option in options_str.split():
            if '=' in option:
                key, value = option.split('=', 1)
                try:
                    info['options'][key] = int(value) if value.isdigit() else float(value)
                except ValueError:
                    info['options'][key] = value
            else:
                info['options'][option] = True

    if parts_str.endswith(' static'):
        info['is_static'] = True
        parts_str = parts_str[:-7].strip()
    elif parts_str.endswith(' pos'):
        info['is_pos'] = True
        parts_str = parts_str[:-4].strip()

    info['path'] = [p.strip() for p in parts_str.split(' - ')]

    return info

def export_layers():
    app = Krita.instance()
    document = app.activeDocument()
    if not document:
        print("No active document.")
        return

    export_dir = "C:\\tmp"
    file_format = 'png'

    document_name = document.fileName() if document.fileName() else 'Untitled'
    file_name, _ = os.path.splitext(os.path.basename(document_name))
    export_path = os.path.join(export_dir, file_name)

    if not os.path.exists(export_path):
        os.makedirs(export_path)

    app.setBatchmode(True)

    layers_data = []
    nodes_to_process = list(document.rootNode().childNodes())

    # First pass: Collect data about all layers
    for node in nodes_to_process:
        if node.type() != 'paintlayer':
            continue

        parsed_info = parse_layer_name(node.name())
        if not parsed_info:
            continue

        bounds = node.bounds()
        center = bounds.center()

        layer_info = {
            'node': node,
            'parsed': parsed_info,
            'bounds': bounds,
            'center': (center.x(), center.y())
        }
        layers_data.append(layer_info)

    # Reverse to process from top layer down, as in Krita's layer stack
    layers_data.reverse()

    processed_layers = {}
    tree_json = []

    # Second pass: Process and export
    for layer_data in layers_data:
        node = layer_data['node']
        parsed = layer_data['parsed']
        bounds = layer_data['bounds']
        center = layer_data['center']
        path_key = ' - '.join(parsed['path'])

        # Handle position layers
        if parsed['is_pos']:
            if path_key not in processed_layers:
                processed_layers[path_key] = {}
            processed_layers[path_key]['pos'] = center
            continue

        if parsed['is_root_pos']:
            if 'root' not in processed_layers:
                processed_layers['root'] = {}
            processed_layers['root']['pos'] = center
            continue

        # Handle static layers
        if parsed['is_static']:
            if path_key not in processed_layers:
                processed_layers[path_key] = {}
            processed_layers[path_key]['static_node'] = node
            continue

        # Handle main layers
        if path_key not in processed_layers:
            processed_layers[path_key] = {}
        
        layer_name = parsed['path'][-1]
        parent_name = parsed['path'][-2] if len(parsed['path']) > 1 else 'root'

        processed_layers[path_key].update({
            'name': layer_name,
            'parent': parent_name,
            'offset': (bounds.x(), bounds.y()),
            'size': (bounds.width(), bounds.height()),
            'options': parsed['options'],
            'node': node,
            'bounds': bounds
        })

    # Third pass: Finalize and export based on the original layer order
    root_entry = {
        'name': 'root',
        'size': (document.width(), document.height())
    }
    if 'root' in processed_layers and 'pos' in processed_layers['root']:
        root_entry['pos'] = processed_layers['root']['pos']
    else:
        root_entry['pos'] = (document.width() // 2, document.height() // 2)
    tree_json.append(root_entry)

    # First handle position-only layers that don't have a main layer
    for path_key, data in processed_layers.items():
        if path_key == 'root' or 'node' in data:
            continue
        
        if 'pos' in data:
            parts = path_key.split(' - ')
            layer_name = parts[-1]
            parent_name = parts[-2] if len(parts) > 1 else 'root'
            
            json_entry = {
                'name': layer_name,
                'parent': parent_name,
                'pos': data['pos']
            }
            tree_json.append(json_entry)

    # Then process main layers
    for layer_data in layers_data:
        parsed = layer_data['parsed']
        # Skip non-main layers, as they are already handled
        if parsed['is_pos'] or parsed['is_static'] or parsed['is_root_pos']:
            continue

        path_key = ' - '.join(parsed['path'])
        data = processed_layers.get(path_key)

        if not data or 'node' not in data:
            continue

        # Export main layer image
        layer_name = data['name']
        file_path = os.path.join(export_path, f"{layer_name}.{file_format}")
        data['node'].save(file_path, 96, 96, krita.InfoObject(), data['bounds'])

        # Export static layer image if it exists
        if 'static_node' in data:
            static_file_path = os.path.join(export_path, f"{layer_name}_static.{file_format}")
            data['static_node'].save(static_file_path, 96, 96, krita.InfoObject(), data['bounds'])

        # Prepare entry for JSON
        json_entry = {
            'name': data['name'],
            'parent': data['parent'],
            'offset': data['offset'],
            'size': data['size']
        }
        if 'pos' in data:
            json_entry['pos'] = data['pos']
        else:
            center = data['bounds'].center()
            json_entry['pos'] = (center.x(), center.y())
        
        json_entry.update(data['options'])
        tree_json.append(json_entry)

    # Save the final JSON file
    json_path = os.path.join(export_path, 'tree.json')
    with open(json_path, 'w') as f:
        json.dump(tree_json, f, indent=4)

    app.setBatchmode(False)
    print(f"Export complete. Files saved to {export_path}")

# Run the export function
export_layers()
